; ****************************************************************************
; "DRVEXCH" - Exchange drive letters (device driver + executable)
;
; Written by Michal H. Tyc <mht@bttr-software.de>
; for BTTR Software
;
; Copyright (C) 1997-2001 BTTR Software
; [Under GNU GENERAL PUBLIC LICENSE]
; Published by BTTR Software
;
; This program is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; This program is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with this program; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
; ****************************************************************************

  section .text

  org 0  ; it is a device driver

; device driver header
devdrvrhdr
  jmp short runcom      ; will be patched to -1 later
  dw -1                 ; pointer to next device driver header
  dw 1000000000000000b  ; device driver attributes (character device driver)
  dw stratfunc          ; offset to device driver strategy routine
  dw intfunc            ; offset to device driver interrupt routine
  db "DRVEXCH$"         ; device driver name

; when loaded as COM, control is passed here
runcom:
  xor cx, cx
  push cs                         ; retf will restore cs = PSP and jump to
  push cx                         ; PSP:0 (int 20h, terminate program)
  mov byte [0ffh], 0dh            ; cut command line against Borland IDE bug
  mov bp, [2]                     ; first unavailable paragraph above program
  mov ax, cs
  mov word [paramptr + 102h], ax
  sub ax, -16                     ; segment fixup, cf = 1
  mov si, .segjmp                 ; (COM program would have "org 256")
  push ax
  push si
  retf                            ; jmp far .segjmp
.segjmp:
  call main                       ; main procedure, with cx = 0, cf = 1
  retf                            ; terminate

; device driver strategy routine
stratfunc:
  mov [cs:reqhdraddr], bx      ; save pointer to device request header
  mov [cs:reqhdraddr + 2], es
  retf

; device driver interrupt routine
intfunc:
  pushf
  push ax
  push cx
  push dx
  push bx
  push bp
  push si
  push di
  push ds
  push es                      ; save registers
  lds bx, [cs:reqhdraddr]
  mov dx, 8103h                ; error code "invalid command"
  cmp byte [bx + 2], 0         ; device command is "initialize" (0)?
  jne .error
  mov bp, [bx + 0eh]           ; offset of first unavailable byte above driver
  mov cl, 4                    ; (DOS 5+ only, otherwise garbage?)
  shr bp, cl
  add bp, [bx + 10h]           ; first unavailable paragraph above the driver
  inc bp                       ; plus one for safety (if offset & 0fh nonzero)
  les ax, [bx + 12h]
  mov [cs:paramptr], ax
  mov [cs:paramptr + 2], es    ; pointer to parameter text
  xor cx, cx                   ; cf = 0
  call main                    ; main procedure, with cx = 0, cf = 0
  or word [0], byte -1         ; header fix (set to 0ffffh)
  lds bx, [reqhdraddr]
  and word [bx + 0eh], byte 0  ; shorter form of zeroing a word
  mov word [bx + 10h], cs      ; no memory reserved
  mov dx, 100h                 ; done
.error:
  mov word [bx + 3], dx  ; set status
  pop es
  pop ds
  pop di
  pop si
  pop bp
  pop bx
  pop dx
  pop cx
  pop ax
  popf                   ; restore registers
  retf

; in both cases (loaded as a COM program and as a device driver, what is
; indicated by carry flag) we arrive here -- to the main procedure
; on entry bp = first free paragraph above the program/driver, cx = 0
main:
  rcr ch, 1            ; set ch = 80h if COM program
  push cx              ; cx changed by DOS version check
  mov dx, intromsg
  call print1          ; display program name and license, set ds = cs
  mov ah, 30h
  int 21h              ; get DOS version
  pop cx
  cmp al, 3
  mov dx, wrongdosmsg
  jb print             ; can't work with DOS < 3, quit
  cmp al, 5
  xchg al, ah          ; swap major and minor bytes
  mov [dosver], ax     ; save DOS version
  jnb .memtest         ; test available memory always under DOS 5+
  test ch, ch          ; but in case of a device driver
  jns .nomemtest       ; it's impossible under DOS 3 and 4
.memtest:
  mov ax, cs
  add ax, totalsizepar + 16  ; end of needed memory plus some stack for COM
  cmp ax, bp
  mov dx, nomemmsg
  ja print1                  ; not enough memory, quit
.nomemtest:
  lds si, [paramptr]
  cld                 ; all strings "up"
  test ch, ch         ; check COM identification flag
  js processpar       ; if loaded as a COM program

; search for the first space after device driver pathname
searchsp:
  lodsb
  cmp al, ' '
  je processpar
  test al, al    ; unexpected end of ASCIIZ string?
  jne searchsp

; subroutines are inserted here for convenience
; (short jumps, etc., and not for clarity)

; print message with or w/o exit
paramerr:
  push dx           ; this entry is from main
paramerr2:
  pop dx            ; and this one from subroutines (pop ret address)
  mov dx, usagemsg

; set ds = cs
print1:
  push cs
  pop ds

; print message at ds:dx
print:
  mov ah, 9
  int 21h
  ret

; skip spaces in the parameter text
skipsp:
  lodsb        ; get next character
  cmp al, ' '
  je skipsp
  ret

; convert drive letter to DOS drive # (A: = 0, etc.)
lettertodrv:
  and al, 0dfh  ; to uppercase
  sub al, 'A'
  jb paramerr2  ; correct are A:
  cmp al, 31
  ja paramerr2  ; to `:  (Z: in older DOSes, but it doesn't matter here)
  ret

; process command line parameters (cl = 0 here, ch <> 0 if COM)
processpar:
  push cs
  pop es            ; explicit swapping data (max. 32 pairs, table overflow
  mov di, swapdata  ; impossible because of command line length limit)
.next:
  call skipsp       ; get first non-space character
  cmp al, 13
  je .done          ; line terminator is CR (COM program)
  test al, al
  je .done          ; or NUL (device driver)
  cmp al, '#'
  jne .norearr
  test cx, cx       ; any drives given (cl <> 0) or COM program (ch <> 0)?
  jne paramerr      ; if so, auto-rearrange not available
  dec cl            ; mark auto-rearrange (cl = -1)
  jmp short .next3
.norearr:
  test cl, cl       ; auto-rearrange already turned on?
  js paramerr       ; if so, explicit swapping not available
  call lettertodrv  ; first drive # in pair
  mov ah, al        ; remember it
  lodsb
  cmp al, '-'       ; correct form is "X-Y"
  jne paramerr
  lodsb
  call lettertodrv  ; second drive # in pair
  inc cx            ; count pairs to swap in cl
  stosw             ; store current pair in swapdata table
.next3:
  lodsb
  cmp al, ' '   ; parameter correctly terminated?
  je .next
  cmp al, 13    ; end of line?
  je .done
  test al, al   ; end of line?
  jne paramerr  ; else wrong syntax
.done:
  test cl, cl
  je paramerr  ; no parameters given
  push cs
  pop ds       ; restore ds = cs

; now a table of DBPs will be prepared
; drives with no DPB will have weight 0
getdpbs:
  mov ah, 0dh
  int 21h                   ; reset drives
  mov ah, 52h
  int 21h                   ; es:bx = List of Lists
  cmp byte [dosver + 1], 4  ; next DPB pointer in DOS 4+ is moved, so
  sbb bp, bp                ; bp = -1 if DOS 3, else 0
  mov si, dpbvecs           ; begin of table
  les di, [es:bx]           ; first DPB
.next:
  mov bl, [es:di]             ; DOS drive #
  mov bh, 0
  inc byte [bx + weights]     ; mark DPB presence
  shl bx, 1
  shl bx, 1                   ; table index for drive #
  mov [si + bx], di
  mov [si + bx + 2], es
  les di, [es:di + bp + 19h]  ; next DPB (equiv. to [es:di + 18h] for DOS 3)
  cmp di, byte -1
  jne .next                   ; ?:0ffffh = end of chain

; what is to be done now?
; cl still holds number of swaps or -1 when no explicit swaps
  test cl, cl
  mov ch, 0     ; cx = counter
  js testhds    ; < 0: re-arrange by BIOS #
  jmp explicit  ; > 0: explicit swaps

; check on which HD each drive is physically located
testhds:
  mov ax, 3513h
  int 21h
  mov [int13hhook.jmpold + 1], bx
  mov [int13hhook.jmpold + 3], es  ; get and save int 13h vector
  mov ah, 25h
  mov dx, int13hhook
  int 21h                          ; hook int 13h
  mov [diskpacket.segm], cs        ; prepare disk address packet
  mov dx, foundmsg
  call print                       ; print header
  mov dl, 2                        ; idendify locations
  mov cl, 30                       ; of all potential 30 drives (C:-`:)
  mov di, weights + 2
.next:
  mov al, 0
  mov byte [int13hdrv], al  ; clear last accessed drive
  cmp [di], al
  je .nothd                 ; drive has no DPB
  push di
  push cx
  push dx
  xchg ax, dx               ; al = drive #
  mov cx, -1
  mov bx, diskpacket        ; address packet (sector and buffer pointer)
  int 25h                   ; read logical sector 0
  jnb .done
  cmp ax, 207h              ; wrong address size? (need FAT32 call?)
  jne .done
  popf                      ; after int 25h
  pop dx
  push dx
  inc dx                    ; dl = drive # plus 1
  mov cx, -1
  mov bx, diskpacket
  xor si, si                ; r/w flags: read
  mov ax, 7305h
  int 21h                   ; read FAT32 logical sector 0
  pushf
.done:
  popf
  pop dx
  pop cx
  pop di
  mov al, byte [int13hdrv]  ; last accessed drive BIOS #
  test al, al
  jns .nothd                ; hard disks are 80h, 81h, ...
  xor al, 0b0h              ; make an ASCII digit from it
  mov [hdmsg2], al          ; prepare message
  mov ah, 32
  mul ah
  or al, dl                 ; drive's weight for sorting =
  mov [di], al              ; = 32 * HD + DOS_drive  (HD is 0 to 7)
  push dx
  mov dx, hdmsg1
  call print                ; print drive info
  pop dx
.nothd:
  inc di
  inc dx             ; next drive
  inc byte [hdmsg1]  ; next message
  loop .next

; end of the loop
  mov ax, 2513h
  lds dx, [int13hhook.jmpold + 1]
  int 21h                          ; restore int 13h vector
  push cs
  pop ds                           ; restore ds

; now re-arrange drive letters according to their weights
; drives with no DPB (weight = 0) can't be swapped, and drives with
; no physical HD detected (weight = 1) will not be re-arranged automatically
sortdrives:
  mov di, weights + 2  ; di points to weights of drives
  mov dl, 2            ; dl = DOS drive # (we start from C:)
.next1
  mov bl, [di]      ; bl = first weight
  cmp bl, 1
  jbe .skip         ; not a HD, don't swap
  mov bh, bl        ; bh = lowest weight so far
  mov ah, 0         ; ah = drive to swap with (0 = none)
  lea si, [di + 1]  ; scan only the rest of the table (drives > dl)
  mov dh, dl        ; dh = drive currently compared with drive dl
.next2:
  inc dh            ; next drive
  cmp dh, 32        ; whole table scanned?
  jnb .swap
  lodsb             ; al = next weight
  cmp al, 1
  jbe .next2        ; no DPB, don't swap
  cmp al, bh
  jnb .next2        ; lower weight already found, next drive
  mov ah, dh        ; so far, drive dl needs swapping with this one
  mov bh, al        ; remember lowest weight found so far
  mov bp, si        ; save weight table index (plus 1)
  jmp short .next2
.swap:
  test ah, ah
  je .skip             ; no swapping needed in this cycle
  xchg [di], bh
  mov [ds:bp - 1], bh  ; swap weights
  mov al, dl
  call swapdrives      ; swap drives al, ah
.skip:
  inc di            ; next weight
  inc dx            ; next drive
  cmp dl, 31        ; all drives (except `: that cannot be
  jb .next1         ; swapped with itself)?
  mov dx, donemsg
  call print
  jmp short finish  ; all done

; swap pairs of drives given explicitly on command line (cx = count)
explicit:
  mov si, swapdata
  mov bh, 0         ; bl will be an index
.next:
  lodsw                   ; al, ah = pair of drives to be swapped
  mov bl, al
  cmp [bx + weights], bh
  je .notswap             ; weight = 0, no DPB, so not swappable
  mov bl, ah
  cmp [bx + weights], bh
  jne .swap               ; swappable
.notswap:
  add ax, 'AA'         ; convert drive numbers to letters
  mov [wrongdrv1], ah
  mov [wrongdrv2], al
  mov dx, wrongdrvmsg
  jmp print            ; print error message and exit
.swap:
  call swapdrives  ; swap drives al, ah
  loop .next

; reset drives and return from main
finish:
  mov ah, 0dh
  int 21h
  ret

; do actual DPB swapping
swapdrives:
  push cx
  push dx
  push bx
  push si
  push di                      ; save registers
  xchg ax, dx
  mov al, 4
  mul dl
  xchg ax, bx
  les di, [bx + dpbvecs]       ; DPB of first drive
  mov al, 4
  mul dh
  xchg ax, bx
  lds si, [bx + dpbvecs]       ; DPB of second drive
  inc di
  inc si                       ; skip DBP first byte (unit number)
  cmp byte [cs:dosver + 1], 4  ; in DOS 4+ DBP first part (below the pointer)
  sbb cx, cx                   ; is longer, so cx = -1 if DOS 3, else 0
  add cx, byte 24              ; cx = length of swapped blocks (23 for DOS 3)
.swap1:
  mov al, [es:di]
  movsb
  mov [si - 1], al            ; swap byte [es:di] with [ds:si]
  loop .swap1                 ; DPB first part: 01h-17h/01h-18h
  lea si, [si + 4]            ;           (DOS:   3.x  /  4+   )
  lea di, [di + 4]            ; skip pointer to next DPB
  mov cl, 2                   ; length of DPB second part in words
  cmp word [cs:dosver], 70ah
  jb .swap2
  mov cl, 16                  ; larger DPB (FAT32, DOS 7.10)
.swap2:
  mov ax, [es:di]
  movsw
  mov [si - 2], ax  ; swap word [es:di] with [ds:si]
  loop .swap2       ; DPB second part: 1ch-1fh/1dh-20h/1dh-3ch
  pop di            ;            (DOS:   3.x  /4.0-7.0/  7.10 )
  pop si
  pop bx
  pop dx
  pop cx
  push cs
  pop ds            ; restore registers
  ret

; int 13h hook to save accessed drive number
int13hhook:
  cmp ah, 2               ; function: read sector
  je .readsec
  cmp ah, 42h             ; function: extended (LBA) read
  jne .jmpold
.readsec:
  mov [cs:int13hdrv], dl  ; save accessed drive BIOS #
.jmpold:
  jmp 0:0                 ; chain to the previous handler

  align 4

; int 25h disk address packet
diskpacket
  dd 0           ; logical sector
  dw 1           ; number of sectors
  dw sectorbuff  ; offset
.segm
  dw 0           ; segment

paramptr
  dw 129, 0  ; PSP command line tail pointer

weights
  times 32 db 0  ; weights of drives for sorting (0 = drive doesn't exist)

; name and license of program
intromsg
  db 'DRVEXCH Version 0.4.2', 13, 10
  db 'Copyright (C) 1997-2001 BTTR Software', 13, 10
  db '[Under GNU GENERAL PUBLIC LICENSE]', 13, 10
  db 'Published by BTTR Software', 13, 10, '$'

; error messages
nomemmsg
  db 13, 10
  db 'Error: Not enough memory.', 13, 10, '$'

usagemsg
  db 13, 10
  db 'Usage: [DEVICE=]DRVEXCH.COM [#|X-Y [W-Z [...]]]', 13, 10
  db 13, 10
  db '  #   = auto-order drive letters by physical HD number (DEVICE only)', 13, 10
  db '  X-Y = swap drive letters X: and Y:', 13, 10, '$'

wrongdosmsg
  db 13, 10
  db 'Error: DOS 3.0 or higher required.', 13, 10, '$'

wrongdrvmsg
  db 13, 10
  db 'Error: Drives '
wrongdrv1
  db '?: and '
wrongdrv2
  db '?: cannot be swapped.', 13, 10, '$'

; other messages
foundmsg
  db 13, 10
  db 'Drives found:', 13, 10, '$'
hdmsg1
  db 'C: on HD'
hdmsg2
  db '0', 13, 10, '$'

donemsg
  db 13, 10
  db 'Re-arranging done.', 13, 10, '$'

  align 4

endofcode:

  section .bss

swapdata
  resw 32  ; pairs of drives to swap

dpbvecs
  resd 32  ; pointers to DPB tables

reqhdraddr
  resd 1  ; pointer to device request header

dosver
  resw 1  ; DOS version

int13hdrv
  resb 1  ; drive accessed by int 13h
  resb 1  ; alignment byte

sectorbuff
  resb 8192  ; maximum sector size, probably too much

endofdata:

totalsizepar equ ((endofcode - devdrvrhdr) + (endofdata - swapdata)) >> 4

  end
